//--------------------------------------------------------------------------- 
// Protocol for Lightweight Authentication of Identity (PLAID)
//
// Cardholder applet
//
// Reference Implementation compliant with AS 5185 - Javacard 2.x source code
// 
// --------------------------------------------------------------------------- 
// This implementation:  Copyright Commonwealth of Australia, 2010.
// PLAID:  Copyright Commonwealth of Australia, 2006-2010.
// 
// No part of PLAID or its source code may be reproduced, digitised, stored in 
// a retrieval system,  communicated to the public or caused to be seen or 
// heard in public, made publicly available or publicly performed, offered for 
// sale or hire or exhibited by way of trade in public or distributed by way 
// of trade in any form or by any means, electronic, mechanical or otherwise 
// without either the written permission of the Commonwealth represented by 
// the Commonwealth Service Delivery Agency (Centrelink) or as licensed under 
// the Software Licence Terms (Licence).
// 
// By using PLAID and/or its source code you agree to be bound by the Licence.
// 
// A copy of the Licence is distributed with this source and/or executable 
// code in the Centrelink document: "Protocol for Lightweight Authentication 
// of Identity (PLAID), LOGICAL SMARTCARD APPLICATION SPECIFICATION"
//
// A copy of this document and Licence is also available upon email request 
// from PLAID@centrelink.gov.au or by download from www.govdex.gov.au 
// ---------------------------------------------------------------------------
// Issue Date: March 2010 
//
// Status: Prototype release 0.800

package plaid8;

import javacard.framework.*;
import javacard.security.*;
import javacardx.crypto.*; 

/**
 * <b>Class: PLAID</b><p>
 * <b>Generic Description</b><p>
 * This javacard applet implements PLAID authentication and the associated 
 * management routines as specified in Australian Standard AS 5185.<p>
 * <b>Design Considerations</b><p>
 * The following has been implemented in this reference model: <ul>
 * <li> There are 8 keypairs (RSA(IA) and AES(FA)) instantiated. This amount is
 * mandated by TOTAL_KEY_SETS.
 * <li> The variable "keyData" manages the correlation between the keySetID
 * values and the corresponding index position.
 * <li> The Shillkeys (decoy keys) are stored in their own cipher objects.
 * <li> The Shillkeys are generated in the constructor.
 * <li> PLAID authentication in Admin mode (using keysetID 0x0000) can only be 
 * performed through the contact interface.
 * <li> Before the applet is secured, the body of a "set data" command is DER
 * encoded.
 * <li> After the applet is secured, the body of a "set data" command is DER
 * encoded and then encrypted using AES.
 * <li> The method "processGetData" is blank in this reference implementation
 * as no additional user fields have been specified.
 */

public class PLAID8 extends Applet
{
  //CLA constants
  static final byte CLA_PROPRIETARY_CMD       =(byte)0x80;
    
  //INS constants  
  static final byte INS_INITIAL_AUTHENTICATE  =(byte)0x8A;
  static final byte INS_FINAL_AUTHENTICATE    =(byte)0x8C;
  static final byte INS_SET_DATA              =(byte)0xDB;
  static final byte INS_GET_DATA              =(byte)0xCB;

  //TAG constants
  static final byte DIVERSIFICATION_DATA      =(byte)0x01;
  static final byte ACS_RECORD                =(byte)0x02;
  static final byte PIN                       =(byte)0x03;
  static final byte SECURE_ICC                =(byte)0x04;
  static final byte MINUTIAE                  =(byte)0x06;
  static final byte IA_KEY                    =(byte)0x07;
  static final byte FA_KEY                    =(byte)0x08;
  static final byte REINITIALISE_CARD         =(byte)0x0A;
    
  //Opmode consts
  static final byte[] OPMODE_1FACTOR          = {0x00,0x00};
  static final byte[] OPMODE_2FACTOR_PIN      = {0x00,0x01};
  static final byte[] OPMODE_2FACTOR_MINUTIAE = {0x00,0x02};
  
  //State constants 
  static final byte STATE_IDLE                =(byte)0x00;
  static final byte STATE_IA_COMPLETED        =(byte)0x01;
  static final byte STATE_FA_COMPLETED        =(byte)0x02;
  static final byte STATE_INITIAL             =(byte)0x00;
  static final byte STATE_SECURED             =(byte)0x01;
    
  //Offset constants - persistant (cardData)
  static final short OFFSET_VERSION               =(short)0;
  static final short OFFSET_SECURITY_STATE        =(short)1;
  static final short OFFSET_DIVERSIFICATION_DATA  =(short)2;
  static final short OFFSET_ACS_RECORD            =(short)10;
  static final short OFFSET_PIN_HASH              =(short)18;
  static final short OFFSET_MINUTIAE              =(short)38;
  
  //Length constants
  static final short LENGTH_VERSION               =(short)1;  
  static final short LENGTH_DIVERSIFICATION_DATA  =(short)8;
  static final short LENGTH_ACS_RECORD            =(short)8;
  static final short LENGTH_KEYSETID              =(short)2;
  static final short LENGTH_PIN_HASH              =(short)20;
  static final short LENGTH_MINUTIAE              =(short)224;
  
  //Offset constants - transient (currentSession)
  static final short OFFSET_CURRENT_STATE         = (short)0;
  static final short OFFSET_KEYSETID              = (short)1;
  static final short OFFSET_OPMODEID              = (short)3;

  //Misc      
  static final byte[] PUBLIC_EXPONENT      = {0x01,0x00,0x01};
  static final short TOTAL_KEY_SETS        = (short)8;
  static final byte OCTET_STRING_ASN1      = (byte)0x04;
  static final byte SEQUENCE_ASN1          = (byte)0x10;
  static final byte[] ADMIN_KEYSETID       = {0x00,0x00};
  static final byte PLAID_VERSION          = (byte)0x08;
  
  //Persistant objects
  private final byte[] cardData          = new byte[268];
  private final byte[] keyData           = new byte[TOTAL_KEY_SETS*2];
  private final RSAPublicKey[] IAKey     = new RSAPublicKey[TOTAL_KEY_SETS];
  private final AESKey[] FAKey           = new AESKey[TOTAL_KEY_SETS];
  private final RSAPublicKey IAShillKey;
  private final AESKey FAShillKey;
  private final AESKey sessionKey;
  private final RandomData rnd;
  private final Cipher AESCipher;
  private final Cipher RSACipher;
  private final MessageDigest SHA1;
                      
  //Transient objects
  private final byte[] currentSession;
  private final short[] keysetIDIndex;
  private final byte[] rnd1Buffer;
  private final byte[] rnd3Buffer;
  private final byte[] Buffer240, Buffer32; 
  
  /**
  * <b>Description</b><p>
  * This method invokes the variables required for the PLAID applet.<p>
  * <b>Design Considerations</b><p>
  * The Shillkeys are generated during applet instantiation.
  * The padding mode used for the RSA object is PKCS#1.5. OAEP padding can
  * be supported however this approach has a significant performance hit
  * and may not be more secure for a protocol that does not expose the
  * RSA modulus.
  */
  private PLAID8() 
  { 
    Util.arrayFillNonAtomic(cardData,(short)0,(short)268,(byte)0x00);
    Util.arrayFillNonAtomic(keyData,(short)0,(short)(TOTAL_KEY_SETS*2),
      (byte)0xFF);
    Buffer240=JCSystem.makeTransientByteArray((short)240, 
      JCSystem.CLEAR_ON_DESELECT);
    rnd1Buffer=JCSystem.makeTransientByteArray((short)16,
      JCSystem.CLEAR_ON_DESELECT);
    rnd3Buffer=JCSystem.makeTransientByteArray((short)16, 
      JCSystem.CLEAR_ON_DESELECT);
    currentSession=JCSystem.makeTransientByteArray((short)5,
      JCSystem.CLEAR_ON_DESELECT);
    keysetIDIndex=JCSystem.makeTransientShortArray((short)1, 
      JCSystem.CLEAR_ON_DESELECT);  
    Buffer32=JCSystem.makeTransientByteArray((short)32, 
      JCSystem.CLEAR_ON_DESELECT);
    rnd=RandomData.getInstance(RandomData.ALG_SECURE_RANDOM);  
    AESCipher=Cipher.getInstance(Cipher.ALG_AES_BLOCK_128_CBC_NOPAD,false);
    RSACipher=Cipher.getInstance(Cipher.ALG_RSA_PKCS1,false);
    SHA1 = MessageDigest.getInstance(MessageDigest.ALG_SHA,false);
    sessionKey=(AESKey)KeyBuilder.buildKey(KeyBuilder.TYPE_AES_TRANSIENT_RESET,
      KeyBuilder.LENGTH_AES_128,false);
    for (short Index=0;Index<TOTAL_KEY_SETS;Index++)
    {
      IAKey[Index]=(RSAPublicKey)KeyBuilder.buildKey(
        KeyBuilder.TYPE_RSA_PUBLIC,KeyBuilder.LENGTH_RSA_1024,false);
      IAKey[Index].setExponent(PUBLIC_EXPONENT,(short)0,(short)3);
      FAKey[Index]=(AESKey)KeyBuilder.buildKey(KeyBuilder.TYPE_AES, 
        KeyBuilder.LENGTH_AES_128,false);
    }
    KeyPair keyPair=new KeyPair(KeyPair.ALG_RSA_CRT,KeyBuilder.LENGTH_RSA_1024);
    keyPair.genKeyPair();
    IAShillKey=((RSAPublicKey) keyPair.getPublic());
    rnd.generateData(rnd1Buffer,(short)0,(short)16);
    FAShillKey=(AESKey) KeyBuilder.buildKey(
      KeyBuilder.TYPE_AES,KeyBuilder.LENGTH_AES_128,false);
    FAShillKey.setKey(rnd1Buffer,(short)0);
  }
  
  /**
  * <b>Description</b><p>
  * This method registers this applet instance with ICC's JCRE.
  */
  public static void install(byte[] params, short offset, byte length) 
    throws ISOException
  {
    (new PLAID8()).register(params,(short)(offset+1),params[offset]);
  }
  
  /**
  * <b>Return Type:</b> boolean<p>
  * <b>Generic Description</b><p>
  * This method returns true iff the currently active communication channel
  * is through a contact interface as determined by the protocol byte.
  */
  private boolean contactInterface(APDU apdu)
  {
    return ((apdu.getProtocol()&0xF0)==0x00);
  }
  
  /**
  * <b>Return Type:</b> byte[]<p>
  * <b>Generic Description</b><p>
  * This method parses the DER encoded ASN1 in to a format that is used by
  * the processSetData method.
  */
  private byte[] parseASN1(byte[] Buff, short Length)
  {
    byte[] outputBuffer = new byte[225]; 
    byte[] ASN1 = new byte[(short)(Length)];  
    Util.arrayCopyNonAtomic(Buff,(short)0,ASN1,(short)0,(short)Length);
    short indexASN1 = 0;
    short indexOutput = 0;
    if (ASN1[indexASN1++] != SEQUENCE_ASN1) 
      ISOException.throwIt(ISO7816.SW_DATA_INVALID);
    if ((ASN1[indexASN1]&0x80)==0x00) 
      indexASN1++;
    else
    {
      indexASN1++; 
      indexASN1++;
    }
    while (indexASN1 < Length)
    {
      if (ASN1[indexASN1++] != OCTET_STRING_ASN1) 
        ISOException.throwIt(ISO7816.SW_DATA_INVALID);
      if ((ASN1[indexASN1]&0x80)==0x00) 
      {
        Util.arrayCopyNonAtomic(ASN1,(short)(indexASN1+1),outputBuffer,
          (short)indexOutput,(short)ASN1[indexASN1]);
        indexOutput+=(short)ASN1[indexASN1];
        indexASN1+=(short)ASN1[indexASN1];
        indexASN1++;
      }
      else
      {
        indexASN1++;
        Util.arrayCopyNonAtomic(ASN1,(short)(indexASN1+1),outputBuffer,
          (short)indexOutput,Util.makeShort((byte)0x00,ASN1[(short)
          (indexASN1)]));
        indexOutput+=Util.makeShort((byte)0x00,ASN1[indexASN1]);
        indexASN1+=Util.makeShort((byte)0x00,ASN1[indexASN1]);
        indexASN1++;
      }
    }
    byte[] realOutputBuffer = new byte[indexOutput];
    Util.arrayCopyNonAtomic(outputBuffer,(short)0,realOutputBuffer,(short)0,
      (short)indexOutput);
    return realOutputBuffer;
  }
  
  /**
  * <b>Description</b><p>
  * This method determines the nature of the APDU as determined by the
  * instruction byte and invokes the corresponding method. 
  */
  public void process(APDU apdu)
  {
    if (selectingApplet())
      return;  
    short Length = apdu.setIncomingAndReceive();
    switch (apdu.getBuffer()[ISO7816.OFFSET_INS])
    {
      case INS_INITIAL_AUTHENTICATE: 
        processInitialAuthenticate(apdu);
        return;  
      case INS_FINAL_AUTHENTICATE:
        processFinalAuthenticate(apdu);
        return;  
      case INS_SET_DATA: 
        processSetData(apdu,Length);
        return;  
      case INS_GET_DATA: 
        processGetData(apdu);
        return;          
      default: 
        ISOException.throwIt(ISO7816.SW_INS_NOT_SUPPORTED);
    }
  }
  
  /**
  * <b>Description</b><p>
  * This method performs the ICC side of Initial Authenticate command as
  * specified in Australian Standard AS-5185<p>
  * <b>Design Considerations</b><p>
  * 1 - The ICC will allow allow authentication in administration mode through
  * the contact interface.<p>
  * 2 - After selecting an RSA key to use, the ICC will continue to "search"
  * through the list of remaining keys to avoid potential timing attacks.<p>
  * 3 - If no appropriate key is found, the ICC will use the Shillkey.<p>
  */
  private void processInitialAuthenticate(APDU apdu)
  {
    byte[] APDUBuffer=apdu.getBuffer();
    short APDUIndex = 9;
    short Iterations = 0;
    if (cardData[OFFSET_SECURITY_STATE] != STATE_SECURED)
      ISOException.throwIt(ISO7816.SW_COMMAND_NOT_ALLOWED);
    while (APDUBuffer[(short)(APDUIndex-1)] == 0x02)
    {
      currentSession[OFFSET_KEYSETID] = APDUBuffer[(short)APDUIndex];
      currentSession[OFFSET_KEYSETID+1] = APDUBuffer[(short)(APDUIndex+1)];
      Iterations = 0;  
      for (byte i=0;i<(TOTAL_KEY_SETS*2);i+=2)
      {
        if ((keyData[i]==currentSession[OFFSET_KEYSETID])&&
          (keyData[(short)(i+1)]==currentSession[OFFSET_KEYSETID+1]))
        {
          if (currentSession[OFFSET_CURRENT_STATE]!=STATE_IA_COMPLETED)
          {
            keysetIDIndex[0] = Iterations;
            RSACipher.init(IAKey[keysetIDIndex[0]],Cipher.MODE_ENCRYPT);  
            currentSession[OFFSET_CURRENT_STATE] = STATE_IA_COMPLETED;
          }
        }  
        Iterations++;
      }
      APDUIndex+=4;
    }
    if (currentSession[OFFSET_CURRENT_STATE] != STATE_IA_COMPLETED)
      RSACipher.init(IAShillKey,Cipher.MODE_ENCRYPT);
    if (Util.arrayCompare(currentSession,OFFSET_KEYSETID,ADMIN_KEYSETID,
      (short)0,(short)2)==0)
    {
      if (!contactInterface(apdu))
        ISOException.throwIt(ISO7816.SW_COMMAND_NOT_ALLOWED);
    }
    Buffer240[0] = keyData[(short)(keysetIDIndex[0]*2)];
    Buffer240[1] = keyData[(short)((keysetIDIndex[0]*2)+1)];
    Util.arrayCopyNonAtomic(cardData,OFFSET_DIVERSIFICATION_DATA,Buffer240,
      (short)2,(short)8);
    rnd.generateData(rnd1Buffer,(short)0,(short)16);
    Util.arrayCopyNonAtomic(rnd1Buffer,(short)0,Buffer240,(short)10,
      (short)16); 
    Util.arrayCopyNonAtomic(rnd1Buffer,(short)0,Buffer240,(short)26,
      (short)16);  
    RSACipher.doFinal(Buffer240,(short)0,(short)42,APDUBuffer,(short)0);
    apdu.setOutgoingAndSend((short)0,(short)128);
  }
  
  /**
  * <b>Description</b><p>
  * This method performs the ICC side of Final Authenticate command as
  * specified in Australian Standard AS-5185<p>
  * <b>Design Considerations</b><p>
  * 1 - In the case where the opmodeID cannot be determined, the ICC
  * responds as if opmodeID was 1-factor. 
  */
  private void processFinalAuthenticate(APDU apdu)
  {
    byte[] APDUBuffer=apdu.getBuffer(); 
    AESCipher.init(FAKey[keysetIDIndex[0]],Cipher.MODE_DECRYPT);
    AESCipher.doFinal(APDUBuffer,ISO7816.OFFSET_CDATA,(short)48,Buffer240,
      (short)0);
    Util.arrayCopyNonAtomic(Buffer240,(short)0,currentSession,OFFSET_OPMODEID,
      (short)2);
    Util.arrayCopyNonAtomic(rnd1Buffer,(short)0,Buffer32,(short)0,(short)16);
    Util.arrayCopyNonAtomic(Buffer240,(short)2,Buffer32,(short)16,(short)16);       
    SHA1.doFinal(Buffer32,(short)0,(short)32,Buffer32,(short)0); 
    Util.arrayCopyNonAtomic(Buffer240,(short)18,rnd3Buffer,(short)0,(short)16);
    sessionKey.setKey(rnd3Buffer,(short)0);
    if (Util.arrayCompare(rnd3Buffer,(short)0, Buffer32,(short)0,(short)16)==0)
    {
      currentSession[OFFSET_CURRENT_STATE] = STATE_FA_COMPLETED;
      AESCipher.init(sessionKey,Cipher.MODE_ENCRYPT);
      Util.arrayCopyNonAtomic(cardData,OFFSET_DIVERSIFICATION_DATA,APDUBuffer,
        (short)0,LENGTH_DIVERSIFICATION_DATA);	      
      Util.arrayCopyNonAtomic(cardData,OFFSET_ACS_RECORD,APDUBuffer,
        LENGTH_DIVERSIFICATION_DATA,LENGTH_ACS_RECORD);
      if (Util.arrayCompare(currentSession,OFFSET_OPMODEID,OPMODE_1FACTOR,
        (short)0,(short)2)==0) 
      {
        AESCipher.doFinal(APDUBuffer,(short)0,(short)16,APDUBuffer,(short)0);
        apdu.setOutgoingAndSend((short)0,(short)16);
        return;
      }
      else if (Util.arrayCompare(currentSession,(short)OFFSET_OPMODEID,
        OPMODE_2FACTOR_PIN,(short)0,(short)2)==0)
      {
        Util.arrayCopyNonAtomic(cardData,OFFSET_PIN_HASH,APDUBuffer,(short)16,
          LENGTH_PIN_HASH);
        Util.arrayFillNonAtomic(APDUBuffer,(short)36,(short)12,(byte)0x00);
        AESCipher.doFinal(APDUBuffer,(short)0,(short)48,APDUBuffer,(short)0);
        apdu.setOutgoingAndSend((short)0,(short)48);
        return;   
      }
      else if (Util.arrayCompare(currentSession,OFFSET_OPMODEID,
        OPMODE_2FACTOR_MINUTIAE,(short)0,(short)2)==0)
      {
        Util.arrayCopyNonAtomic(cardData,OFFSET_MINUTIAE,APDUBuffer,(short)16,
          LENGTH_MINUTIAE);   
        AESCipher.doFinal(APDUBuffer,(short)0,(short)240,APDUBuffer,(short)0);
        apdu.setOutgoingAndSend((short)0,(short)240);
        return;   
      }
    }
    else
    {
      AESCipher.init(FAShillKey,Cipher.MODE_ENCRYPT);
      Util.arrayCopyNonAtomic(rnd1Buffer,(short)0,APDUBuffer,(short)0,
        (short)8);
      Util.arrayCopyNonAtomic(rnd1Buffer,(short)0,APDUBuffer,(short)8,
        (short)8);
      if (Util.arrayCompare(currentSession,OFFSET_OPMODEID,OPMODE_2FACTOR_PIN,
        (short)0,(short)2)==0)
      {
        Util.arrayCopyNonAtomic(rnd1Buffer,(short)0,APDUBuffer,(short)16,
          (short)16);
        Util.arrayFillNonAtomic(APDUBuffer,(short)32,(short)16,(byte)0x00);
        AESCipher.doFinal(APDUBuffer,(short)0,(short)48,APDUBuffer,(short)0);
        apdu.setOutgoingAndSend((short)0,(short)48);
        return;   
      }
      else if (Util.arrayCompare(currentSession,OFFSET_OPMODEID,
        OPMODE_2FACTOR_MINUTIAE,(short)0,(short)2)==0)
      {
        Util.arrayCopyNonAtomic(rnd1Buffer,(short)0,APDUBuffer,(short)16,(short)16);   
        Util.arrayFillNonAtomic(APDUBuffer,(short)16,(short)112,(byte)0xAA);
        Util.arrayFillNonAtomic(APDUBuffer,(short)128,(short)112,(byte)0x55);
        AESCipher.doFinal(APDUBuffer,(short)0,(short)248,APDUBuffer,(short)0);
        apdu.setOutgoingAndSend((short)0,(short)248);
        return;   
      }
      else
      {
        AESCipher.doFinal(APDUBuffer,(short)0,(short)16,APDUBuffer,(short)0);
        apdu.setOutgoingAndSend((short)0,(short)16);
        return;
      }
    }
  }
    
  /**
  * <b>Description</b><p>
  * This method manages the getting of PLAID variables.<p>
  * <b>Design Considerations</b><p>
  * This method is not implemented as all variables in this reference
  * implementation are accessable via the opmodeID. 
  */
  
  private void processGetData(APDU apdu)
  {
    return;    
  }
  
  /**
  * <b>Description</b><p>
  * This method manages the setting of PLAID variables.<p>
  * <b>Design Considerations</b><p>
  * Before a card is secured, variables may be set by storing the value in
  * the body of an APDU in a DER format. Once a card is secured, data can 
  * only be set by completing PLAID authentication in administrator mode and 
  * encrypting the DER encoded body of the APDU with the session key.
  * The setting of the keys requires placement in next available space as 
  * indicated by the array "keyData". In the case where all key storage has
  * been exhusted, the applet will respond with SW_FILE_FULL.
  */
  
  private void processSetData(APDU apdu, short Length)
  {
    byte[] APDUBuffer=apdu.getBuffer();  
    if (!contactInterface(apdu))
      ISOException.throwIt(ISO7816.SW_COMMAND_NOT_ALLOWED);
    if ((currentSession[OFFSET_CURRENT_STATE]==STATE_FA_COMPLETED)&&
      (Util.arrayCompare(currentSession,OFFSET_KEYSETID,ADMIN_KEYSETID,
      (short)0,(short)2)==0))
    {
      AESCipher.init(sessionKey,Cipher.MODE_DECRYPT);
      AESCipher.doFinal(APDUBuffer,ISO7816.OFFSET_CDATA,(short)Length,
        Buffer240,(short)0);
    }
    else if (cardData[OFFSET_SECURITY_STATE]==STATE_INITIAL)
    {
      Util.arrayCopyNonAtomic(APDUBuffer,ISO7816.OFFSET_CDATA,Buffer240,
        (short)0,(short)Length);
    }
    else
    {
      ISOException.throwIt(ISO7816.SW_COMMAND_NOT_ALLOWED);
    }
    short Index;
    byte[] newKeysetID = new byte[2];
    byte[] parsedASN1 = parseASN1(Buffer240,Length);
    switch (parsedASN1[(short)0x00])
    {
      case DIVERSIFICATION_DATA:
        Util.arrayCopy(parsedASN1,(short)1, cardData,(short)
          OFFSET_DIVERSIFICATION_DATA,(short)LENGTH_DIVERSIFICATION_DATA);
        return;
      case ACS_RECORD:
        Util.arrayCopy(parsedASN1,(short)1, cardData,(short)
          OFFSET_ACS_RECORD,(short)LENGTH_ACS_RECORD);
        return;
      case MINUTIAE:
        Util.arrayCopy(parsedASN1,(short)1, cardData, (short)
          OFFSET_MINUTIAE,(short)LENGTH_MINUTIAE);
        return;
      case PIN:
        SHA1.reset();
        SHA1.doFinal(parsedASN1,(short)1,(short)8,cardData,
          (short)OFFSET_PIN_HASH);
        return;
      case SECURE_ICC:
        cardData[OFFSET_SECURITY_STATE] = STATE_SECURED;
        return;
      case REINITIALISE_CARD:
        Util.arrayFillNonAtomic(cardData,(short)0,(short)268,(byte)0x00);
        Util.arrayFillNonAtomic(keyData,(short)0,(short)(TOTAL_KEY_SETS*2),
          (byte)0xFF);
        cardData[OFFSET_SECURITY_STATE] = STATE_INITIAL;
        return;
      case IA_KEY:
        Util.arrayCopy(parsedASN1,(short)1,newKeysetID,(short)0,(short)2);
        byte[] IAKeyArr = new byte[128];
        Util.arrayCopy(parsedASN1,(short)3,IAKeyArr,(short)0,(short)128);
        if ((newKeysetID[0] == ADMIN_KEYSETID[0])&&(newKeysetID[1] == 
          ADMIN_KEYSETID[1]))
        {
          keyData[0] = ADMIN_KEYSETID[0];
          keyData[1] = ADMIN_KEYSETID[1];
          IAKey[0].setModulus(IAKeyArr,(short)0,(short)128); 
          return; 
        }
        Index = 1;
        for (byte i=2;i<(TOTAL_KEY_SETS*2);i+=2)
        {
          if ((keyData[i]==newKeysetID[0])&&(keyData[(short)(i+1)]==
            newKeysetID[1]))
          {
            IAKey[Index].setModulus(IAKeyArr,(short)0,(short)128); 
            return;
          }  
          Index++;
        }
        for (byte i=1;i<TOTAL_KEY_SETS;i++)
        {
          if ((keyData[(short)(i*2)]==(byte)0xFF)&(keyData[(short)((i*2)+1)]==
            (byte)0xFF))
          {
            IAKey[i].setModulus(IAKeyArr,(short)0,(short)128); 
            keyData[(short)(i*2)] = newKeysetID[0];
            keyData[(short)((i*2)+1)] = newKeysetID[1];
            return;
          }
        }
        ISOException.throwIt((short)ISO7816.SW_FILE_FULL);
        return;
      case FA_KEY:
        Util.arrayCopy(parsedASN1,(short)1,newKeysetID,(short)0,(short)2);
        byte[] FAKeyArr = new byte[16];
        Util.arrayCopy(parsedASN1,(short)3,FAKeyArr,(short)0,(short)16);
        if ((newKeysetID[0] == ADMIN_KEYSETID[0])&&(newKeysetID[1] == 
          ADMIN_KEYSETID[1]))
        {
          keyData[0] = ADMIN_KEYSETID[0];
          keyData[1] = ADMIN_KEYSETID[1];
          FAKey[0].setKey(FAKeyArr,(short)0); 
          return; 
        }
        Index = 1;
        for (byte i=2;i<(TOTAL_KEY_SETS*2);i+=2)
        {
          if ((keyData[i]==newKeysetID[0])&&(keyData[(short)(i+1)]==
            newKeysetID[1]))
          {
            FAKey[Index].setKey(FAKeyArr,(short)0); 
            return;
          }  
          Index++;
        }
    }
  }
    
}